/**
* \file: errmemd_backend_tcpip.c
*
* Implementation of the tcpip backend.
* 
* This file implements the interface functions provided by the tcpip
* backend. A client can connect to this daemon using a socket interface.
* The necessary functionality doing this is implemented here.
*
* \component: errmemd
*
* \author: Kai Tomerius (ktomerius@de.adit-jv.com)
*          Markus Kretschmann (mkretschmann@de.adit-jv.com)
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
* <history item>
*/

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include "errmem_backend.h"

/*
 * Implementation of an error memory backend that dumps error memory
 * messages over TCP/IP.
 */

// the error memory backend
typedef struct ErrmemBackendTcpip {
	ErrmemBackend_t backend; // error memory backend interface
	ErrmemBackendBuffered_t* buffered;
	int fd;                  // the TCP/IP connection
} ErrmemBackendTcpip_t;

// store - log an error memory message
static unsigned store(struct ErrmemBackend* backend, int len,
		  struct errmem_message* msg)
{
	ErrmemBackendTcpip_t* l = (ErrmemBackendTcpip_t*) backend;
	l->buffered->backend.store(&l->buffered->backend, len, msg);
	write(l->fd, l->buffered->buffer, strlen(l->buffered->buffer));
	return msg->internal.seqnum;
}

// destroy - free resources
static void destroy(struct ErrmemBackend* backend)
{
	ErrmemBackendTcpip_t* l = (ErrmemBackendTcpip_t*) backend;

	l->buffered->backend.destroy(&l->buffered->backend);
	free(l);
}

static ErrmemBackendTcpip_t* create()
{
    ErrmemBackendTcpip_t* l =
	calloc(1, sizeof(ErrmemBackendTcpip_t));

    if (l) {
	// setup the error memory backend interface
	l->backend.store = store;
	l->backend.destroy = destroy;

	ErrmemBackendDev_t p_dev = {.type = BUFFER};
	l->buffered = (ErrmemBackendBuffered_t*)
		errmem_backend_create_buffered(&p_dev);
    }
    return l;
}

// errmem_backend_create_tcpip - create a TCP/IP error memory backend
ErrmemBackend_t* errmem_backend_create_tcpip(ErrmemBackend_t* next, int fd)
{
    ErrmemBackendTcpip_t* l = create();
    if (!l || !l->buffered)
	return NULL;
    
    l->backend.next = next;
    l->fd = fd;

    return &l->backend;
}
